<?php
declare(strict_types=1);

namespace App\Controller\Admin;

use App\Controller\AppController;

/**
 * Cars Controller
 *
 * @property \App\Model\Table\CarsTable $Cars
 */
class CarsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Cars->find()->orderByDesc('Cars.created');
        $cars = $this->paginate($query);

        $this->set(compact('cars'));
    }

    /**
     * View method
     *
     * @param string|null $id Car id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $car = $this->Cars->get($id, contain: ['CarParts']);
        $this->set(compact('car'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $car = $this->Cars->newEmptyEntity();
        if ($this->request->is('post')) {
            $car = $this->Cars->patchEntity($car, $this->request->getData());
            if ($this->Cars->save($car)) {
                $this->Flash->success(__('The car has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The car could not be saved. Please, try again.'));
        }
        $this->set(compact('car'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Car id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $car = $this->Cars->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $car = $this->Cars->patchEntity($car, $this->request->getData());
            if ($this->Cars->save($car)) {
                $this->Flash->success(__('The car has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The car could not be saved. Please, try again.'));
        }
        $this->set(compact('car'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Car id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $car = $this->Cars->get($id);
        if ($this->Cars->delete($car)) {
            $this->Flash->success(__('The car has been deleted.'));
        } else {
            $this->Flash->error(__('The car could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function csv()
    {
        $cars = $this->Cars
            ->find()
            ->contain(['CarParts' => ['Titles', 'Countries']])
            ->all();

        $header = [
            'Date',
            'Payment',
            'Team name',
            'Car theme',
            'Contact mobile',
            'Heard about',
            'Heard about other',
            'Amount',
            'Paid',
            'Tx. Status',
            'Tx. Detail',
            'Tx. code',
            'SagePay ID',
            'Card type',
            'Card Last digits',
            'Title',
            'Title Other',
            'Name',
            'Surname',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Country',
            'Telephone',
            'Mobile phone',
            'Email',
            'Contact by email',
            'Contact by phone',
            'Contact by letter',
            'Title',
            'Title Other',
            'Name',
            'Surname',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Country',
            'Telephone',
            'Mobile phone',
            'Email',
            'Can contact by email',
            'Can contact by phone',
            'Can contact by letter',
        ];

        // header('Content-Type: text/csv; charset=utf-8');
        // header('Content-Disposition: attachment; filename=cars-' . date('Y-m-d-h-i-s') . '.csv');
        $csv = fopen('php://temp/maxmemory:' . 5 * 1024 * 1024, 'r+');

        fputcsv($csv, $header);

        foreach ($cars as $reg) {
            $row = [
                $reg->created->format('Y-m-d H:i:s'),
                $reg->payment,
                $reg->team_name,
                $reg->car_theme,
                $reg->contact_mobile,
                $reg->heard_about,
                $reg->heard_about_other,
                $reg->amount,
                $reg->paid ? 'Yes' : 'No',
                $reg->tx_status,
                $reg->tx_status_detail,
                $reg->tx_code,
                $reg->tx_sagepay_id,
                $reg->tx_cardtype,
                $reg->tx_card_lastdigits,
            ];
            foreach ($reg->car_parts as $p) {
                $row = array_merge($row, [
                    $p->title->name,
                    $p->title_other,
                    $p->first_name,
                    $p->surname,
                    $p->address_1,
                    $p->address_2,
                    $p->city,
                    $p->county,
                    $p->postal_code,
                    $p->country->name,
                    $p->telephone,
                    $p->mobile_phone,
                    $p->email,
                    $p->optin_email ? 'Yes' : 'No',
                    $p->optin_telephone ? 'Yes' : 'No',
                    $p->optin_letter ? 'Yes' : 'No',
                ]);
            }

            fputcsv($csv, $row);
        }

        rewind($csv);
        $output = stream_get_contents($csv);

        $response = $this->response;
        $response = $response->withStringBody($output);
        $response = $response->withType('csv');
        $response = $response->withDownload('cars-' . date('Y-m-d-h-i-s') . '.csv');
        return $response;
    }
}
