<?php
declare(strict_types=1);

namespace App\Controller\Admin;

use App\Controller\AppController;

/**
 * Donations Controller
 *
 * @property \App\Model\Table\DonationsTable $Donations
 */
class DonationsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Donations->find()->contain(['Titles', 'Countries']);
        $donations = $this->paginate($query, ['order' => ['Donations.created' => 'desc']]);

        $this->set(compact('donations'));
    }

    /**
     * View method
     *
     * @param string|null $id Donation id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $donation = $this->Donations->get($id, contain: ['Titles', 'Countries']);
        $this->set(compact('donation'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $donation = $this->Donations->newEmptyEntity();
        if ($this->request->is('post')) {
            $donation = $this->Donations->patchEntity($donation, $this->request->getData());
            if ($this->Donations->save($donation)) {
                $this->Flash->success(__('The donation has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The donation could not be saved. Please, try again.'));
        }
        $titles = $this->Donations->Titles->find('list', limit: 200)->all();
        $countries = $this->Donations->Countries->find('list', limit: 200)->all();
        $this->set(compact('donation', 'titles', 'countries'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Donation id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $donation = $this->Donations->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $donation = $this->Donations->patchEntity($donation, $this->request->getData());
            if ($this->Donations->save($donation)) {
                $this->Flash->success(__('The donation has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The donation could not be saved. Please, try again.'));
        }
        $titles = $this->Donations->Titles->find('list', limit: 200)->all();
        $countries = $this->Donations->Countries->find('list', limit: 200)->all();
        $this->set(compact('donation', 'titles', 'countries'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Donation id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $donation = $this->Donations->get($id);
        if ($this->Donations->delete($donation)) {
            $this->Flash->success(__('The donation has been deleted.'));
        } else {
            $this->Flash->error(__('The donation could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function csv()
    {
        $donations = $this->Donations->find()->contain(['Titles', 'Countries']);

        if (!empty($this->request->getQuery('event_id'))) {
            $donations = $donations->where(['Donations.event_id' => $this->request->getQuery('event_id')]);
        }

        $header = [
            'Date',
            'Amount',
            'Understood Gift Aid',
            'Gift Aid',
            'Title',
            'Title Other',
            'Name',
            'Surname',
            'Organisation',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Country',
            'Telephone',
            'Mobile phone',
            'Email',
            'Can contact by email',
            'Can contact by phone',
            'Can contact by letter',
            'Message',
            'Tx. Status',
            'Tx. Detail',
            'Tx. code',
            'SagePay ID',
            'Card type',
            'Card Last digits',
        ];

        // header('Content-Type: text/csv; charset=utf-8');
        // header('Content-Disposition: attachment; filename=donations-' . date('Y-m-d-h-i-s') . '.csv');
        $csv = fopen('php://temp/maxmemory:' . 5 * 1024 * 1024, 'r+');

        fputcsv($csv, $header);

        foreach ($donations as $reg) {
            $row = [
                $reg->created->format('Y-m-d H:i:s'),
                $reg->donation_amount,
                $reg->gift_aid_understood ? 'Yes' : 'No',
                $reg->gift_aid_for,
                $reg->title->name,
                $reg->title_other,
                $reg->first_name,
                $reg->surname,
                $reg->organisation,
                $reg->address_1,
                $reg->address_2,
                $reg->city,
                $reg->county,
                $reg->postal_code,
                $reg->country->name,
                $reg->telephone,
                $reg->mobile_phone,
                $reg->email,
                $reg->optin_email ? 'Yes' : 'No',
                $reg->optin_telephone ? 'Yes' : 'No',
                $reg->optin_letter ? 'Yes' : 'No',
                $reg->message,
                $reg->tx_status,
                $reg->tx_status_detail,
                $reg->tx_code,
                $reg->tx_sagepay_id,
                $reg->tx_cardtype,
                $reg->tx_card_lastdigits,
            ];

            fputcsv($csv, $row);
        }

        rewind($csv);
        $output = stream_get_contents($csv);

        $response = $this->response;
        $response = $response->withStringBody($output);
        $response = $response->withType('csv');
        $response = $response->withDownload('donations-' . date('Y-m-d-h-i-s') . '.csv');
        return $response;
    }
}
