<?php
declare(strict_types=1);

namespace App\Controller\Admin;

use App\Controller\AppController;

/**
 * Registrations Controller
 *
 * @property \App\Model\Table\RegistrationsTable $Registrations
 */
class RegistrationsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Registrations
            ->find()
            ->orderByDesc('Registrations.id')
            ->contain(['Titles', 'Countries', 'Events', 'Tickets' => ['EventCosts']]);

        if (!empty($this->request->getQuery('event_id'))) {
            $query = $query->where(['Registrations.event_id' => $this->request->getQuery('event_id')]);
        }

        $registrations = $this->paginate($query);

        $events = $this->Registrations->Events->find('list', limit: 200)->orderByDesc('id');

        $this->set(compact('registrations', 'events'));
    }

    /**
     * View method
     *
     * @param string|null $id Registration id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $registration = $this->Registrations->get(
            $id,
            contain: ['Titles', 'Countries', 'Events', 'Tickets' => ['EventCosts']],
        );
        $this->set(compact('registration'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $registration = $this->Registrations->newEmptyEntity();
        if ($this->request->is('post')) {
            $registration = $this->Registrations->patchEntity($registration, $this->request->getData());
            if ($this->Registrations->save($registration)) {
                $this->Flash->success(__('The registration has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The registration could not be saved. Please, try again.'));
        }
        $titles = $this->Registrations->Titles->find('list', limit: 200)->all();
        $countries = $this->Registrations->Countries->find('list', limit: 200)->all();
        $events = $this->Registrations->Events->find('list', limit: 200)->all();
        $eventCosts = $this->Registrations->EventCosts->find('list', limit: 200)->all();
        $this->set(compact('registration', 'titles', 'countries', 'events', 'eventCosts'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Registration id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $registration = $this->Registrations->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $registration = $this->Registrations->patchEntity($registration, $this->request->getData());
            if ($this->Registrations->save($registration)) {
                $this->Flash->success(__('The registration has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The registration could not be saved. Please, try again.'));
        }
        $titles = $this->Registrations->Titles->find('list', limit: 200)->all();
        $countries = $this->Registrations->Countries->find('list', limit: 200)->all();
        $events = $this->Registrations->Events->find('list', limit: 200)->all();
        $eventCosts = $this->Registrations->EventCosts->find('list', limit: 200)->all();
        $this->set(compact('registration', 'titles', 'countries', 'events', 'eventCosts'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Registration id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $registration = $this->Registrations->get($id);
        if ($this->Registrations->delete($registration)) {
            $this->Flash->success(__('The registration has been deleted.'));
        } else {
            $this->Flash->error(__('The registration could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function csv()
    {
        $registrations = $this->Registrations
            ->find()
            ->contain(['Titles', 'Countries', 'Events', 'Tickets' => ['EventCosts']]);

        if (!empty($this->request->getQuery('event_id'))) {
            $registrations = $registrations->where(['Registrations.event_id' => $this->request->getQuery('event_id')]);
        }

        $header = [
            'Date',
            'Event',
            'Total cost',
            'Title',
            'Title other',
            'Name',
            'Surname',
            'DOB',
            'Organisation',
            'Address 1',
            'Address2',
            'City',
            'County',
            'Postal Code',
            'Email',
            'Telephone',
            'Mobile',
            'Contact by email',
            'Contact by telephone',
            'Contact by letter',
            'Heard about',
            'Heard about other',
            'Emergency Contact',
            'Emergency Contact No.',
            'Medical conditions',
            'Custom field 1',
            'Custom field 2',
            'Tx. Status',
            'Tx. Detail',
            'Tx. code',
            'SagePay ID',
            'Card type',
            'Card Last digits',
            'Ticket type',
            'Price',
            '# ordered',
            'Subtotal',
        ];

        // header('Content-Type: text/csv; charset=utf-8');
        // header('Content-Disposition: attachment; filename=registrations-' . date('Y-m-d-h-i-s') . '.csv');
        $csv = fopen('php://temp/maxmemory:' . 5 * 1024 * 1024, 'r+');

        fputcsv($csv, $header);

        foreach ($registrations as $reg) {
            $row = [
                $reg->created->format('Y-m-d H:i:s'),
                $reg->event->name,
                $reg->amount,
                $reg->title->name,
                $reg->title_other,
                $reg->first_name,
                $reg->surname,
                $reg->date_of_birth,
                $reg->organisation,
                $reg->address_1,
                $reg->address_2,
                $reg->city,
                $reg->county,
                $reg->postal_code,
                $reg->email,
                $reg->telephone,
                $reg->mobile,
                $reg->optin_email ? 'Yes' : 'No',
                $reg->optin_telephone ? 'Yes' : 'No',
                $reg->optin_letter ? 'Yes' : 'No',
                $reg->heard_about,
                $reg->heard_about_other,
                $reg->emergency_contact,
                $reg->emergency_contact_number,
                $reg->medical_conditions,
                $reg->custom_field_1,
                $reg->custom_field_2,
                $reg->tx_status,
                $reg->tx_status_detail,
                $reg->tx_code,
                $reg->tx_sagepay_id,
                $reg->tx_cardtype,
                $reg->tx_card_lastdigits,
            ];

            foreach ($reg->tickets as $ticket) {
                $row[] = $ticket->event_cost->name;
                $row[] = $ticket->event_cost->price;
                $row[] = $ticket->amount;
                $row[] = $ticket->subTotal();
            }
            fputcsv($csv, $row);
        }

        rewind($csv);
        $output = stream_get_contents($csv);

        $response = $this->response;
        $response = $response->withStringBody($output);
        $response = $response->withType('csv');
        $response = $response->withDownload('registrations-' . date('Y-m-d-h-i-s') . '.csv');
        return $response;
    }
}
