<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Cars Controller
 *
 * @property \App\Model\Table\CarsTable $Cars
 */
class CarsController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    /**
     * View method
     *
     * @param string|null $id Car id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function confirm()
    {
        // $this->loadComponent('SagePay');

        if (!$this->request->getSession()->check('car_id')) {
            return $this->redirect(['action' => 'register']);
        }

        $car = $this->Cars->get(
            $this->request->getSession()->read('car_id'),
            contain: ['CarParts' => ['Countries', 'Titles']],
        );
        $sagepay_url = Configure::read('Forms.SagePay.url');
        $paylater_url = Router::url(['controller' => 'cars', 'action' => 'pay_later'], true);

        // $crypted = $this->SagePay->buildCrypt($car);

        $crypted = $this->buildCrypt($car);
        $this->set(compact('car', 'crypted', 'sagepay_url', 'paylater_url'));
    }

    public function payLater()
    {
        if (!$this->request->getSession()->check('car_id')) {
            return $this->redirect(['action' => 'register']);
        }

        $car = $this->Cars->get(
            $this->request->getSession()->read('car_id'),
            contain: ['CarParts' => ['Countries', 'Titles']],
        );

        if (empty($car->is_mail_sent)) {
            $this->_emailConfirmation($car);
            $this->_emailConfirmation($car, true);

            $car->is_mail_sent = true;
            $car->paid = false;
            $this->Cars->save($car);
            $this->request->getSession()->consume('car_id');
        }
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function register()
    {
        $this->loadComponent('Recaptcha.Recaptcha');
        $validation_error = false;

        $car = $this->Cars->newEmptyEntity();
        $car->car_amount = $this->request->getQuery('amount');
        $car->gift_aid_understood = $this->request->getQuery('gift_aid_understood');

        if ($this->request->getSession()->check('car_id') && !$validation_error) {
            $car = $this->Cars->get(
                $this->request->getSession()->read('car_id'),
                contain: ['CarParts' => ['Titles', 'Countries']],
            );
        }

        if ($this->request->is(['post', 'put'])) {
            $car = $this->Cars->patchEntity($car, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                if (count($car->car_parts) > $car->team_number) {
                    $car->car_parts = array_slice($car->car_parts, 0, $car->team_number);
                }
                if ($this->Cars->save($car)) {
                    $this->request->getSession()->write('car_id', $car->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(__('The entry could not be saved. Please, try again.'));
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        // debug($car);

        $titles = $this->Cars->CarParts->Titles->find('list')->all();
        $countries = $this->Cars->CarParts->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $gift_aid_options = Configure::read('Forms.gift_aid_options');

        $google_captcha_key = Configure::read('GoogleCaptcha.key');
        $car_settings = $this->getTableLocator()
            ->get('CarSettings')
            ->find()
            ->first();

        $this->set(compact('car', 'titles', 'countries', 'gift_aid_options', 'google_captcha_key', 'car_settings'));
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('car_id')) {
            $car = $this->Cars->get(
                $this->request->getSession()->read('car_id'),
                contain: ['CarParts' => ['Titles', 'Countries']],
            );
        } else {
            return $this->redirect(['action' => 'register']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $car->tx_status = $decoded['Status'];
        $car->tx_status_detail = $decoded['StatusDetail'];
        $car->tx_code = $decoded['VendorTxCode'];
        $car->tx_sagepay_id = $decoded['VPSTxId'];
        $car->tx_cardtype = $decoded['CardType'];
        $car->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $car->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $car->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $car->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Cars->save($car);
        $this->set(compact('car', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($car->is_mail_sent)) {
            $this->_emailConfirmation($car);
            $this->_emailConfirmation($car, true);

            $car->is_mail_sent = true;
            $car->paid = true;
            $this->Cars->save($car);
            $this->request->getSession()->consume('car_id');
        }
    }

    private function _emailConfirmation($car, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo($admin ? Configure::read('Forms.confirmation_email_recipients.cars') : $car->car_parts[0]->email)
            ->setSubject(Configure::read('Forms.Email.subject.cars') . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'car' => $car,
            ])
            ->viewBuilder()
            ->setTemplate('car_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($car)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'cars', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $d = $car->car_parts[0];

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount(number_format(400, 2));
        $sagePay->setDescription(Configure::read('Forms.SagePay.item_description.cars'));
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.cars'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
