<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Donations Controller
 *
 * @property \App\Model\Table\DonationsTable $Donations
 */
class DonationsController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    /**
     * View method
     *
     * @param string|null $id Donation id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function confirm()
    {
        // $this->loadComponent('SagePay');

        if (!$this->request->getSession()->check('donation_id')) {
            return $this->redirect(['action' => 'donate']);
        }

        $donation = $this->Donations->get(
            $this->request->getSession()->read('donation_id'),
            contain: ['Titles', 'Countries'],
        );
        $sagepay_url = Configure::read('Forms.SagePay.url');

        // $crypted = $this->SagePay->buildCrypt($donation);

        $crypted = $this->buildCrypt($donation);
        $this->set(compact('donation', 'crypted', 'sagepay_url'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function donate()
    {
        $this->loadComponent('Recaptcha.Recaptcha');
        $validation_error = false;

        $donation = $this->Donations->newEmptyEntity();
        $donation->donation_amount = $this->request->getQuery('amount');
        $donation->gift_aid_understood = $this->request->getQuery('gift_aid_understood');

        if ($this->request->getSession()->check('donation_id') && !$validation_error) {
            $donation = $this->Donations->get(
                $this->request->getSession()->read('donation_id'),
                contain: ['Titles', 'Countries'],
            );
        }

        if ($this->request->is(['post', 'put'])) {
            $donation = $this->Donations->patchEntity($donation, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                if ($this->Donations->save($donation)) {
                    // $this->Flash->success(__('The donation has been saved.'));
                    $this->request->getSession()->write('donation_id', $donation->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(__('The donation could not be saved. Please, try again.'));
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        $titles = $this->Donations->Titles->find('list')->all();
        $countries = $this->Donations->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $gift_aid_options = Configure::read('Forms.gift_aid_options');

        $google_captcha_key = Configure::read('GoogleCaptcha.key');

        $amounts = $this->fetchTable('Amounts')
            ->find()
            ->all();

        $this->set(compact('donation', 'titles', 'countries', 'gift_aid_options', 'google_captcha_key', 'amounts'));
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('donation_id')) {
            $donation = $this->Donations->get(
                $this->request->getSession()->read('donation_id'),
                contain: ['Titles', 'Countries'],
            );
        } else {
            return $this->redirect(['action' => 'donate']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $donation->tx_status = $decoded['Status'];
        $donation->tx_status_detail = $decoded['StatusDetail'];
        $donation->tx_code = $decoded['VendorTxCode'];
        $donation->tx_sagepay_id = $decoded['VPSTxId'];
        $donation->tx_cardtype = $decoded['CardType'];
        $donation->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $donation->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $donation->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $donation->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Donations->save($donation);
        $this->set(compact('donation', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($donation->is_mail_sent)) {
            $this->_emailConfirmation($donation);
            $this->_emailConfirmation($donation, true);

            $donation->is_mail_sent = true;
            $this->Donations->save($donation);
            $this->request->getSession()->consume('donation_id');
        }
    }

    private function _emailConfirmation($donation, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo($admin ? Configure::read('Forms.confirmation_email_recipients.donations') : $donation->email)
            ->setSubject(Configure::read('Forms.Email.subject.donations') . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'donation' => $donation,
            ])
            ->viewBuilder()
            ->setTemplate('donation_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($d)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'donations', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount($d->donation_amount);
        $sagePay->setDescription(Configure::read('Forms.SagePay.item_description.donations'));
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.donations'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
