<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Golfs Controller
 *
 * @property \App\Model\Table\GolfsTable $Golfs
 */
class GolfsController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    /**
     * View method
     *
     * @param string|null $id Golf id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function confirm()
    {
        // $this->loadComponent('SagePay');

        if (!$this->request->getSession()->check('golf_id')) {
            return $this->redirect(['action' => 'register']);
        }

        $golf = $this->Golfs->get(
            $this->request->getSession()->read('golf_id'),
            contain: ['GolfTypes', 'GolfPlayers' => ['Countries', 'Titles']],
        );
        $sagepay_url = Configure::read('Forms.SagePay.url');

        // $crypted = $this->SagePay->buildCrypt($golf);

        $crypted = $this->buildCrypt($golf);
        $this->set(compact('golf', 'crypted', 'sagepay_url'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function register()
    {
        $this->loadComponent('Recaptcha.Recaptcha');

        $validation_error = false;

        $golf = $this->Golfs->newEmptyEntity();
        $golf->golf_amount = $this->request->getQuery('amount');
        $golf->gift_aid_understood = $this->request->getQuery('gift_aid_understood');

        if ($this->request->getSession()->check('golf_id') && !$validation_error) {
            $golf = $this->Golfs->get(
                $this->request->getSession()->read('golf_id'),
                contain: ['GolfPlayers' => ['Titles', 'Countries'], 'GolfTypes'],
            );
        }

        $gtypes = [];
        $types = $this->Golfs->GolfTypes->find('all');

        foreach ($types as $t) {
            $gtypes[$t->id] = $t->players;
        }

        if ($this->request->is(['post', 'put'])) {
            $golf = $this->Golfs->patchEntity($golf, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                if ($this->Golfs->save($golf)) {
                    // $this->Flash->success(__('The golf has been saved.'));
                    $this->request->getSession()->write('golf_id', $golf->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(
                    __('Your registration could not be saved. Please check below for errors and try again.'),
                );
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        $titles = $this->Golfs->GolfPlayers->Titles->find('list')->all();
        $countries = $this->Golfs->GolfPlayers->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $gift_aid_options = Configure::read('Forms.gift_aid_options');

        $google_captcha_key = Configure::read('GoogleCaptcha.key');

        $golf_settings = $this->getTableLocator()
            ->get('GolfSettings')
            ->find()
            ->first();

        $golf_types = $this->Golfs->GolfTypes->find('list');
        $sponsorships = $this->getTableLocator()
            ->get('GolfSponsorships')
            ->find('list');

        $this->set(
            compact(
                'golf',
                'titles',
                'countries',
                'gift_aid_options',
                'google_captcha_key',
                'golf_types',
                'gtypes',
                'golf_settings',
                'sponsorships',
            ),
        );
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('golf_id')) {
            $golf = $this->Golfs->get(
                $this->request->getSession()->read('golf_id'),
                contain: ['GolfPlayers' => ['Titles', 'Countries'], 'GolfTypes'],
            );
        } else {
            return $this->redirect(['action' => 'register']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $golf->tx_status = $decoded['Status'];
        $golf->tx_status_detail = $decoded['StatusDetail'];
        $golf->tx_code = $decoded['VendorTxCode'];
        $golf->tx_sagepay_id = $decoded['VPSTxId'];
        $golf->tx_cardtype = $decoded['CardType'];
        $golf->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $golf->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $golf->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $golf->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Golfs->save($golf);
        $this->set(compact('golf', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($golf->is_mail_sent)) {
            $this->_emailConfirmation($golf);
            $this->_emailConfirmation($golf, true);

            $golf->is_mail_sent = true;
            $golf->paid = true;
            $golf->amount = $decoded['Amount'];
            $this->Golfs->save($golf);
            $this->request->getSession()->consume('golf_id');
        }
    }

    private function _emailConfirmation($golf, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo(
                $admin ? Configure::read('Forms.confirmation_email_recipients.golfs') : $golf->golf_players[0]->email,
            )
            ->setSubject(Configure::read('Forms.Email.subject.golfs') . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'golf' => $golf,
            ])
            ->viewBuilder()
            ->setTemplate('golf_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($golf)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'golfs', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $d = $golf->golf_players[0];

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount($golf->golf_type->price);
        $sagePay->setDescription(Configure::read('Forms.SagePay.item_description.golfs'));
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.golfs'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
