<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;
use Cake\Event\EventInterface;

/**
 * Orders Controller
 *
 * @property \App\Model\Table\OrdersTable $Orders
 */
class OrdersController extends AppController
{
    public function beforeFilter(EventInterface $event)
    {
        Configure::load('payments');

        parent::beforeFilter($event);
    }

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function shop()
    {
        $this->loadComponent('Recaptcha.Recaptcha');
        $validation_error = false;
        $order = $this->Orders->newEmptyEntity();

        if ($this->request->getSession()->check('order_id') && !$validation_error) {
            $order = $this->Orders->get(
                $this->request->getSession()->read('order_id'),
                contain: ['Titles', 'Countries', 'OrderItems'],
            );
        }

        if ($this->request->is(['post', 'put'])) {
            $order = $this->Orders->patchEntity($order, $this->request->getData());
            if ($this->Recaptcha->verify()) {
                foreach ($order->order_items as $item) {
                    $item->price = $this->fetchTable('Products')->get($item->product_id)->price;
                }
                $order->amount = $order->grandTotal();

                if ($this->Orders->save($order)) {
                    // $this->Flash->success(__('The order has been saved.'));
                    $this->request->getSession()->write('order_id', $order->id);
                    return $this->redirect(['action' => 'confirm']);
                }
                $validation_error = true;
                $this->Flash->error(__('The order could not be saved. Please, try again.'));
            } else {
                $this->Flash->error(__('Google Recaptcha validation failed. Please, try again.'));
            }
        }

        $products = $this->fetchTable('Products')
            ->find('all')
            ->where(['active' => true]);

        $google_captcha_key = '';

        $titles = $this->Orders->Titles->find('list')->all();
        $countries = $this->Orders->Countries
            ->find('list')
            ->orderByAsc('name')
            ->all();

        $this->set(compact('order', 'products', 'google_captcha_key', 'titles', 'countries'));
    }

    /**
     * View method
     *
     * @param string|null $id Order id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function confirm()
    {
        $order = $this->Orders->get(
            $this->request->getSession()->read('order_id'),
            contain: ['Titles', 'DeliveryTitles', 'Countries', 'DeliveryCountries', 'OrderItems' => ['Products']],
        );

        $sagepay_url = Configure::read('Forms.SagePay.url');
        $crypted = $this->buildCrypt($order);

        $this->set(compact('order', 'crypted', 'sagepay_url'));
    }

    function transactionResult()
    {
        if ($this->request->getSession()->check('order_id')) {
            $order = $this->Orders->get(
                $this->request->getSession()->read('order_id'),
                contain: ['Titles', 'DeliveryTitles', 'Countries', 'DeliveryCountries', 'OrderItems' => ['Products']],
            );
        } else {
            return $this->redirect(['action' => 'shop']);
        }

        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $strCrypt = $this->request->getQuery('crypt');

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $decoded = $sagePay->decode($strCrypt);

        $order->tx_status = $decoded['Status'];
        $order->tx_status_detail = $decoded['StatusDetail'];
        $order->tx_code = $decoded['VendorTxCode'];
        $order->tx_sagepay_id = $decoded['VPSTxId'];
        $order->tx_cardtype = $decoded['CardType'];
        $order->decline_code = !empty($decoded['DeclineCode']) ? $decoded['DeclineCode'] : '';
        $order->expiry_date = !empty($decoded['ExpiryDate']) ? $decoded['ExpiryDate'] : '';
        $order->bank_auth_code = !empty($decoded['BankAuthCode']) ? $decoded['BankAuthCode'] : '';
        $order->tx_card_lastdigits = $decoded['Last4Digits'];

        $this->Orders->save($order);
        $this->set(compact('order', 'decoded'));

        if ($decoded['Status'] == 'OK' && empty($order->is_mail_sent)) {
            $this->_emailConfirmation($order);
            $this->_emailConfirmation($order, true);

            $order->is_mail_sent = true;
            $order->paid = true;
            $this->Orders->save($order);
            $this->request->getSession()->consume('order_id');
        }
    }

    private function _emailConfirmation($order, $admin = null)
    {
        $email = new Mailer('default');
        $email
            ->setFrom(Configure::read('Forms.SagePay.from_email'))
            ->setTo($admin ? Configure::read('Forms.confirmation_email_recipients.orders') : $order->email)
            ->setSubject(Configure::read('Forms.Email.subject.orders') . ($admin ? ' (admin)' : ''))
            ->setEmailFormat('html')
            ->setViewVars([
                'order' => $order,
            ])
            ->viewBuilder()
            ->setTemplate('order_confirmation')
            ->setLayout('default');

        $email->send();
    }

    private function buildCrypt($d)
    {
        require_once 'sagepay_lib.php';

        $trans_mode = Configure::read('Forms.SagePay.trans_mode');
        $encryption = Configure::read('Forms.SagePay.encryption');

        $url = Router::url(['controller' => 'orders', 'action' => 'transaction_result'], true);

        $strVendorTxCode = Configure::read('Forms.SagePay.vendor_name') . rand(0, 32000) * rand(0, 32000);

        $sagePay = new \SagePay();
        $sagePay->setEncryptPassword($encryption);
        $sagePay->setVendorTxCode($strVendorTxCode);
        $sagePay->setCurrency(Configure::read('Forms.SagePay.currency'));
        $sagePay->setAmount($d->grandTotal());
        $sagePay->setDescription(Configure::read('Forms.SagePay.item_description.orders'));
        $sagePay->setCustomerName($d->first_name . ' ' . $d->surname);
        $sagePay->setSendEMail('1');
        $sagePay->setCustomerEMail($d->email);
        $sagePay->setVendorEMail(Configure::read('Forms.SagePay.vendor_email'));
        $sagePay->setEMailMessage(Configure::read('Forms.Email.message.orders'));
        $sagePay->setBillingFirstnames($d->first_name);
        $sagePay->setBillingSurname($d->surname);
        $sagePay->setBillingAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setBillingCity($d->city);
        $sagePay->setBillingPhone($d->mobile_phone);
        $sagePay->setBillingPostCode($trans_mode == 'live' ? $d->postal_code : '412');
        $sagePay->setBillingCountry($d->country->iso_code);
        $sagePay->setDeliveryPhone($d->mobile_phone);
        $sagePay->setAllowGiftAid(1);
        $sagePay->setDeliveryFirstnames($d->first_name);
        $sagePay->setDeliverySurname($d->surname);
        $sagePay->setDeliveryAddress1($trans_mode == 'live' ? $d->address_1 : '88');
        $sagePay->setDeliveryCity($d->city);
        $sagePay->setDeliveryCountry($d->country->iso_code);

        if ($d->country->name == 'United States') {
            $sagePay->setBillingState($d->state_code);
        }

        $sagePay->setDeliverySameAsBilling();
        $sagePay->setSuccessURL($url);
        $sagePay->setFailureURL($url);

        return $sagePay->getCrypt();
    }
}
