<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Registrations Model
 *
 * @property \App\Model\Table\TitlesTable&\Cake\ORM\Association\BelongsTo $Titles
 * @property \App\Model\Table\CountriesTable&\Cake\ORM\Association\BelongsTo $Countries
 * @property \App\Model\Table\EventsTable&\Cake\ORM\Association\BelongsTo $Events
 * @property \App\Model\Table\EventCostsTable&\Cake\ORM\Association\BelongsTo $EventCosts
 * @property \App\Model\Table\TicketsTable&\Cake\ORM\Association\HasMany $Tickets
 *
 * @method \App\Model\Entity\Registration newEmptyEntity()
 * @method \App\Model\Entity\Registration newEntity(array $data, array $options = [])
 * @method \App\Model\Entity\Registration[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Registration get($primaryKey, $options = [])
 * @method \App\Model\Entity\Registration findOrCreate($search, ?callable $callback = null, $options = [])
 * @method \App\Model\Entity\Registration patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\Registration[] patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Registration|false save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Registration saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Registration[]|\Cake\Datasource\ResultSetInterface|false saveMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Registration[]|\Cake\Datasource\ResultSetInterface saveManyOrFail(iterable $entities, $options = [])
 * @method \App\Model\Entity\Registration[]|\Cake\Datasource\ResultSetInterface|false deleteMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Registration[]|\Cake\Datasource\ResultSetInterface deleteManyOrFail(iterable $entities, $options = [])
 *
 * @mixin \Cake\ORM\Behavior\TimestampBehavior
 */
class RegistrationsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('registrations');
        $this->setDisplayField('age_on_race');
        $this->setPrimaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Titles', [
            'foreignKey' => 'title_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Countries', [
            'foreignKey' => 'country_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Events', [
            'foreignKey' => 'event_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('EventCosts', [
            'foreignKey' => 'event_cost_id',
        ]);
        $this->hasMany('Tickets', [
            'foreignKey' => 'registration_id',
            'saveStrategy' => 'replace',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator->date('date_of_birth')->allowEmptyDate('date_of_birth');

        $validator->integer('title_id')->notEmptyString('title_id');

        $validator
            ->scalar('first_name')
            ->maxLength('first_name', 255)
            ->requirePresence('first_name', 'create')
            ->notEmptyString('first_name');

        $validator
            ->scalar('surname')
            ->maxLength('surname', 255)
            ->requirePresence('surname', 'create')
            ->notEmptyString('surname');

        $validator
            ->scalar('address_1')
            ->maxLength('address_1', 255)
            ->requirePresence('address_1', 'create')
            ->notEmptyString('address_1');

        $validator
            ->scalar('city')
            ->maxLength('city', 255)
            ->requirePresence('city', 'create')
            ->notEmptyString('city');

        $validator
            ->scalar('county')
            ->maxLength('county', 255)
            ->requirePresence('county', 'create')
            ->notEmptyString('county');

        $validator
            ->scalar('postal_code')
            ->maxLength('postal_code', 255)
            ->requirePresence('postal_code', 'create')
            ->notEmptyString('postal_code');

        $validator->integer('country_id')->notEmptyString('country_id');

        $validator
            ->email('email')
            ->requirePresence('email', 'create')
            ->notEmptyString('email');

        $validator
            ->scalar('email_confirm')
            ->maxLength('email_confirm', 255)
            ->requirePresence('email_confirm', 'create')
            ->notEmptyString('email_confirm');

        $validator->add('email_confirm', 'compareWith', [
            'rule' => ['compareWith', 'email'],
            'message' => 'Email addresses do not match',
        ]);

        $validator
            ->scalar('telephone')
            ->maxLength('telephone', 255)
            ->requirePresence('telephone', 'create')
            ->notEmptyString('telephone');

        $validator
            ->scalar('emergency_contact')
            ->maxLength('emergency_contact', 255)
            ->requirePresence('emergency_contact', 'create')
            ->notEmptyString('emergency_contact');

        $validator
            ->scalar('emergency_contact_number')
            ->maxLength('emergency_contact_number', 255)
            ->requirePresence('emergency_contact_number', 'create')
            ->notEmptyString('emergency_contact_number');

        $validator->scalar('medical_conditions')->allowEmptyString('medical_conditions');

        $validator
            ->boolean('disclaimer')
            ->requirePresence('disclaimer', 'create')
            ->notEmptyString('disclaimer');

        $validator->boolean('is_mail_sent')->notEmptyString('is_mail_sent');

        $validator
            ->scalar('decline_code')
            ->maxLength('decline_code', 255)
            ->allowEmptyString('decline_code');

        $validator
            ->scalar('expiry_date')
            ->maxLength('expiry_date', 255)
            ->allowEmptyString('expiry_date');

        $validator
            ->scalar('bank_auth_code')
            ->maxLength('bank_auth_code', 255)
            ->allowEmptyString('bank_auth_code');

        $validator->integer('event_id')->notEmptyString('event_id');

        $validator->integer('event_cost_id')->allowEmptyString('event_cost_id');

        $validator
            ->scalar('custom_field_1')
            ->maxLength('custom_field_1', 400)
            ->requirePresence('custom_field_1', 'create')
            ->notEmptyString('custom_field_1');

        $validator
            ->scalar('custom_field_2')
            ->maxLength('custom_field_2', 400)
            ->requirePresence('custom_field_2', 'create')
            ->notEmptyString('custom_field_2');

        $validator->boolean('no_email')->notEmptyString('no_email');

        $validator->boolean('no_telephone')->notEmptyString('no_telephone');

        $validator->boolean('no_letter')->notEmptyString('no_letter');

        $validator->boolean('optin_email')->notEmptyString('optin_email');

        $validator->boolean('optin_telephone')->notEmptyString('optin_telephone');

        $validator->boolean('optin_letter')->notEmptyString('optin_letter');

        $validator->scalar('additional_guests')->allowEmptyString('additional_guests');

        $validator
            ->scalar('heard_about')
            ->maxLength('heard_about', 255)
            ->requirePresence('heard_about', 'create')
            ->notEmptyString('heard_about');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn('title_id', 'Titles'), ['errorField' => 'title_id']);
        $rules->add($rules->existsIn('country_id', 'Countries'), ['errorField' => 'country_id']);
        $rules->add($rules->existsIn('event_id', 'Events'), ['errorField' => 'event_id']);
        $rules->add($rules->existsIn('event_cost_id', 'EventCosts'), ['errorField' => 'event_cost_id']);

        return $rules;
    }
}
